<?php

namespace App\Http\Controllers;

use App\Models\ProjectStatusSetting;
use App\Models\Role;
use App\Models\User;
use App\Helper\Files;
use App\Helper\Reply;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Project;
use App\Models\BaseModel;
use App\Models\ClientNote;
use App\Scopes\ActiveScope;
use App\Traits\ImportExcel;
use App\Models\Notification;
use App\Models\ContractType;
use Illuminate\Http\Request;
use App\Imports\ClientImport;
use App\Jobs\ImportClientJob;
use App\Models\ClientDetails;
use App\Models\ClientCategory;
use App\Models\PurposeConsent;
use App\Models\LanguageSetting;
use App\Models\UniversalSearch;
use App\Models\ClientSubCategory;
use App\Models\PurposeConsentUser;
use Illuminate\Support\Facades\DB;
use App\DataTables\TicketDataTable;
use App\DataTables\ClientsDataTable;
use App\DataTables\InvoicesDataTable;
use App\DataTables\PaymentsDataTable;
use App\DataTables\ProjectsDataTable;
use App\DataTables\EstimatesDataTable;
use App\DataTables\ClientGDPRDataTable;
use App\DataTables\ClientNotesDataTable;
use App\DataTables\CreditNotesDataTable;
use App\DataTables\ClientContactsDataTable;
use App\DataTables\OrdersDataTable;
use App\Enums\Salutation;
use App\Http\Requests\Admin\Employee\ImportRequest;
use App\Http\Requests\Admin\Client\StoreClientRequest;
use App\Http\Requests\Gdpr\SaveConsentUserDataRequest;
use App\Http\Requests\Admin\Client\UpdateClientRequest;
use App\Http\Requests\Admin\Employee\ImportProcessRequest;
use App\Models\Lead;
use App\Traits\EmployeeActivityTrait;
use App\Models\ManufacturingEnquiry;
use App\Models\ManufacturingLead;
use App\Models\ManufacturingProposal;
use App\Models\ManufacturingProject;
use App\Models\ManufacturingInventory;
use App\Models\ManufacturingDispatch;
use App\Models\PurchaseRequest;
use App\Models\Allocation;
use App\Models\InventoryAllocation;
use App\Models\SynthesisStage;
use App\Models\StageComment;
use App\Models\ManagerApproval;
use App\Models\RouteOfSynthesis;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use App\Models\ClientDispatchInfo;
use App\Models\OrderShippingDetails;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;


class ManufacturingController extends AccountBaseController
{

    use ImportExcel;
    use EmployeeActivityTrait;

    public function __construct()
    {
        parent::__construct();
        $this->pageTitle = 'Manufacturing';
        $this->middleware(function ($request, $next) {
            abort_403(!in_array('clients', $this->user->modules));

            $this->viewOverviewDashboard = user()->permission('view_overview_dashboard');
            $this->viewProjectDashboard = user()->permission('view_project_dashboard');
            $this->viewClientDashboard = user()->permission('view_client_dashboard');
            $this->viewHRDashboard = user()->permission('view_hr_dashboard');
            $this->viewTicketDashboard = user()->permission('view_ticket_dashboard');
            $this->viewFinanceDashboard = user()->permission('view_finance_dashboard');


            return $next($request);
        });
    }

    /**
     * client list
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, ClientsDataTable $dataTable)
    {
        // Check if the request has a 'data' query parameter
        if ($request->has('data') && $request->input('data') == 'true') {
            return $this->passingData();  // Call the passingData method
        }

        // Otherwise, execute the normal index method logic
        $viewPermission = user()->permission('view_clients');
        $this->addClientPermission = user()->permission('add_clients');
        abort_403(!in_array($viewPermission, ['all', 'added', 'both']));

        if (!request()->ajax()) {
            $this->clients = User::allClients(active: false);
            $this->subcategories = ClientSubCategory::all();
            $this->categories = ClientCategory::all();
            $this->projects = Project::all();
            $this->contracts = ContractType::all();
            $this->countries = countries();
            $this->totalClients = count($this->clients);
            $this->enquiries = ManufacturingEnquiry::all();
            $this->leads = ManufacturingLead::all();
            $this->proposals = ManufacturingProposal::all();
            $this->projects = ManufacturingProject::all();
            $this->inventories = ManufacturingInventory::all();
            $this->dispatches = ManufacturingDispatch::all();
        }

        return $dataTable->render('manufacturing.index', $this->data);
    }

    public function managerDashboard(Request $request, ClientsDataTable $dataTable)
    {
        // Check if the request has a 'data' query parameter
        if ($request->has('data') && $request->input('data') == 'true') {
            return $this->passingData();  // Call the passingData method
        }

        // Otherwise, execute the normal index method logic
        $viewPermission = user()->permission('view_clients');
        $this->addClientPermission = user()->permission('add_clients');
        abort_403(!in_array($viewPermission, ['all', 'added', 'both']));

        if (!request()->ajax()) {
            $this->clients = User::allClients(active: false);
            $this->subcategories = ClientSubCategory::all();
            $this->categories = ClientCategory::all();
            $this->projects = Project::all();
            $this->contracts = ContractType::all();
            $this->countries = countries();
            $this->totalClients = count($this->clients);
            $this->enquiries = ManufacturingEnquiry::all();
            $this->leads = ManufacturingLead::all();
            $this->proposals = ManufacturingProposal::all();
            $this->projects = ManufacturingProject::all();
            $this->inventories = ManufacturingInventory::all();
            $this->dispatches = ManufacturingDispatch::all();
        }

        return $dataTable->render('manufacturing.manager-dashboard', $this->data);
    }


    public function getEnquiries(Request $request)
    {
        try {
            $query = ManufacturingEnquiry::query();

            // Apply filters if they are present
            if ($request->has('client_name') && $request->client_name) {
                $query->where('client_name', 'like', '%' . $request->client_name . '%');
            }

            if ($request->has('project_name') && $request->project_name) {
                $query->where('project_name', 'like', '%' . $request->project_name . '%');
            }

            if ($request->has('status') && $request->status) {
                $query->where('status', $request->status);
            }

            // Apply sorting
            if ($request->has('order')) {
                $columnIndex = $request->input('order.0.column');
                $columnName = $request->input('columns.' . $columnIndex . '.data');
                $direction = $request->input('order.0.dir');
                $query->orderBy($columnName, $direction);
            }

            // Pagination
            $length = $request->input('length', 25); // Default length 25
            $start = $request->input('start', 0); // Default start 0
            $enquiries = $query->offset($start)->limit($length)->get();

            // Return the response in the expected DataTable format
            return response()->json([
                'draw' => $request->draw,
                'recordsTotal' => ManufacturingEnquiry::count(),
                'recordsFiltered' => $query->count(),
                'data' => $enquiries
            ]);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error($e->getMessage());

            // Return an error response
            return response()->json(['error' => 'An error occurred while fetching data.'], 500);
        }
    }

    public function getLeads(Request $request)
    {
        try {
            // Start with a query on ManufacturingLead
            $query = ManufacturingLead::query();

            // Apply filters if they are present
            if ($request->has('lead_name_filter') && $request->lead_name_filter) {
                $query->where('lead_name', $request->lead_name_filter);
            }

            if ($request->has('lead_status_filter') && $request->lead_status_filter) {
                $query->where('lead_status', $request->lead_status_filter);
            }

            if ($request->has('start_date') && $request->start_date) {
                $query->whereDate('created_at', '>=', $request->start_date);
            }

            if ($request->has('end_date') && $request->end_date) {
                $query->whereDate('created_at', '<=', $request->end_date);
            }

            // Apply sorting if requested
            if ($request->has('order')) {
                $columnIndex = $request->input('order.0.column');
                $columnName = $request->input('columns.' . $columnIndex . '.data');
                $direction = $request->input('order.0.dir');
                $query->orderBy($columnName, $direction);
            }

            // Join with the `enquiries` table to get `enquiry_id` and `client_name`
            $query->join('manufacturing_enquiries', 'manufacturing_leads.enquiry_id', '=', 'manufacturing_enquiries.id')
                ->select(
                    'manufacturing_leads.id',
                    'manufacturing_leads.enquiry_id',
                    'manufacturing_leads.lead_name',
                    'manufacturing_leads.lead_status',
                    'manufacturing_leads.estimated_cost',
                    'manufacturing_leads.created_at',
                    'manufacturing_leads.updated_at',
                    'manufacturing_enquiries.client_name'
                );

            // Pagination - getting the correct page of results
            $length = $request->input('length', 25); // Default to 25 if length not provided
            $start = $request->input('start', 0); // Default to 0 if start not provided
            $leads = $query->offset($start)->limit($length)->get();

            // Return the data in a format suitable for DataTable
            return response()->json([
                'draw' => $request->input('draw'),
                'recordsTotal' => ManufacturingLead::count(), // Total number of leads
                'recordsFiltered' => $query->count(), // Number of filtered leads after applying the filters
                'data' => $leads
            ]);
        } catch (\Exception $e) {
            // Log the exception for debugging
            \Log::error($e->getMessage());

            // Return a generic error response
            return response()->json(['error' => 'An error occurred while fetching data.'], 500);
        }
    }


    public function getLeadById($id)
    {
        try {
            // Fetch the lead along with its related enquiry
            $lead = DB::table('manufacturing_leads')
                ->join('manufacturing_enquiries', 'manufacturing_leads.enquiry_id', '=', 'manufacturing_enquiries.id')
                ->select(
                    'manufacturing_leads.id',
                    'manufacturing_leads.enquiry_id',
                    'manufacturing_leads.lead_name',
                    'manufacturing_leads.lead_status',
                    'manufacturing_leads.estimated_cost',
                    'manufacturing_leads.created_at',
                    'manufacturing_leads.updated_at',
                    'manufacturing_enquiries.client_name'
                )
                ->where('manufacturing_leads.id', $id)
                ->first();

            if (!$lead) {
                return response()->json(['error' => 'Lead not found.'], 404);
            }

            // Fetch all enquiries for the dropdown
            $enquiries = DB::table('manufacturing_enquiries')
                ->select('id', 'client_name')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $lead,
                'enquiries' => $enquiries
            ]);
        } catch (\Exception $e) {
            \Log::error('Error fetching lead: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching the lead data.'], 500);
        }
    }

    public function updateLead(Request $request)
    {
        try {
            // Validate incoming data
            $validated = $request->validate([
                'id' => 'required|exists:manufacturing_leads,id',  // Ensure the lead ID exists
                'enquiry_id' => 'required|exists:manufacturing_enquiries,id',  // Ensure the enquiry ID exists
                'lead_name' => 'required|string|max:255',
                'lead_status' => 'required|string',
                'estimated_cost' => 'nullable|numeric',
            ]);

            // Log the validated request to ensure we're receiving the correct data
            \Log::info('Updating lead with data:', $validated);

            // Find the lead by ID
            $lead = ManufacturingLead::find($validated['id']);

            if (!$lead) {
                // If lead is not found, log the error and return a 404 response
                \Log::error('Lead not found with ID: ' . $validated['id']);
                return response()->json(['error' => 'Lead not found.'], 404);
            }

            // Update the lead data
            $lead->enquiry_id = $validated['enquiry_id'];
            $lead->lead_name = $validated['lead_name'];
            $lead->lead_status = $validated['lead_status'];
            $lead->estimated_cost = $validated['estimated_cost'];

            // Save the updated lead
            $lead->save();

            // Log success
            \Log::info('Lead updated successfully with ID: ' . $lead->id);

            return response()->json([
                'success' => true,
                'message' => 'Lead updated successfully',
                'data' => $lead
            ]);
        } catch (\Exception $e) {
            // Log full exception details for debugging
            \Log::error('Error updating lead: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            // Return a generic error response to the client
            return response()->json(['error' => 'An error occurred while updating the lead.'], 500);
        }
    }



    public function storeLeads(Request $request)
    {
        try {
            $lead = new ManufacturingLead();
            $lead->enquiry_id = $request->enquiry_id;
            $lead->lead_name = $request->lead_name;
            $lead->lead_status = $request->lead_status;
            $lead->estimated_cost = $request->estimated_cost;
            $lead->save();

            return response()->json(['message' => 'Lead created successfully!', 'data' => $lead], 200);
        } catch (\Exception $e) {
            \Log::error($e->getMessage());
            return response()->json(['error' => 'An error occurred while creating the lead.'], 500);
        }
    }



    public function getEnquiriesById($id)
    {
        // If an ID is passed, fetch a specific enquiry
        if ($id) {
            $enquiry = ManufacturingEnquiry::find($id);

            if (!$enquiry) {
                return response()->json(['message' => 'Enquiry not found.'], 404);
            }

            return response()->json([
                'data' => [$enquiry],
            ]);
        }
    }

    // Method to store a new enquiry (Add)
    public function storeEnquiries(Request $request)
    {
        // Validate the incoming request
        $validated = $request->validate([
            'client_name' => 'required|string|max:255',
            'project_name' => 'required|string|max:255',
            'description' => 'required|string',
            'status' => 'required|in:active,inactive',
        ]);

        // Create the new enquiry
        $enquiry = ManufacturingEnquiry::create([
            'client_name' => $validated['client_name'],
            'project_name' => $validated['project_name'],
            'description' => $validated['description'],
            'status' => $validated['status'],
        ]);

        return response()->json(['message' => 'Enquiry created successfully.']);
    }

    // Method to update an existing enquiry
    public function updateEnquiry(Request $request)
    {
        try {
            // Validate incoming data
            $validated = $request->validate([
                'client_name' => 'required|string|max:255',
                'project_name' => 'required|string|max:255',
                'description' => 'required|string',
                'status' => 'required|in:active,inactive',
                'enquiry_id' => 'required|exists:manufacturing_enquiries,id',  // Ensure the ID exists in the database
            ]);

            // Find the enquiry by ID
            $enquiry = ManufacturingEnquiry::find($request->enquiry_id);

            if (!$enquiry) {
                return response()->json(['message' => 'Enquiry not found.'], 404);
            }

            // Update the enquiry record
            $enquiry->update([
                'client_name' => $validated['client_name'],
                'project_name' => $validated['project_name'],
                'description' => $validated['description'],
                'status' => $validated['status'],
            ]);

            return response()->json(['message' => 'Enquiry updated successfully.']);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error updating enquiry: ' . $e->getMessage());
            return response()->json(['message' => 'Server Error'], 500);
        }
    }


    public function getProposals(Request $request)
    {
        try {
            // Start with a query on ManufacturingProposal
            $query = ManufacturingProposal::query();

            // Apply filters if they are present
            if ($request->has('proposal_number') && $request->proposal_number) {
                $query->where('proposal_number', 'like', '%' . $request->proposal_number . '%');
            }

            if ($request->has('lead_name_filter') && $request->lead_name_filter) {
                // Assuming there's a relation between 'ManufacturingProposal' and 'ManufacturingLead' 
                // and you want to filter based on lead name (you can adjust according to your relations)
                $query->whereHas('lead', function ($q) use ($request) {
                    $q->where('lead_name', 'like', '%' . $request->lead_name_filter . '%');
                });
            }

            if ($request->has('status_filter') && $request->status_filter) {
                $query->where('status', $request->status_filter);
            }

            if ($request->has('start_date') && $request->start_date) {
                $query->whereDate('created_at', '>=', $request->start_date);
            }

            if ($request->has('end_date') && $request->end_date) {
                $query->whereDate('created_at', '<=', $request->end_date);
            }

            // Apply sorting if requested
            if ($request->has('order')) {
                $columnIndex = $request->input('order.0.column');
                $columnName = $request->input('columns.' . $columnIndex . '.data');
                $direction = $request->input('order.0.dir');
                $query->orderBy($columnName, $direction);
            }

            // Join with the `manufacturing_leads` table to get `lead_id` and `lead_name`
            $query->join('manufacturing_leads', 'manufacturing_proposals.lead_id', '=', 'manufacturing_leads.id')
                ->select(
                    'manufacturing_proposals.id',
                    'manufacturing_proposals.lead_id',
                    'manufacturing_proposals.proposal_number',
                    'manufacturing_proposals.proposal_details',
                    'manufacturing_proposals.total_amount',
                    'manufacturing_proposals.status',
                    'manufacturing_proposals.proposal_status',
                    'manufacturing_proposals.created_at',
                    'manufacturing_proposals.updated_at',
                    'manufacturing_leads.lead_name' // Assuming this is the field to fetch
                );

            // Pagination - getting the correct page of results
            $length = $request->input('length', 25); // Default to 25 if length not provided
            $start = $request->input('start', 0); // Default to 0 if start not provided
            $proposals = $query->offset($start)->limit($length)->get();

            // Return the data in a format suitable for DataTable
            return response()->json([
                'draw' => $request->input('draw'),
                'recordsTotal' => ManufacturingProposal::count(), // Total number of proposals
                'recordsFiltered' => $query->count(), // Number of filtered proposals after applying the filters
                'data' => $proposals
            ]);
        } catch (\Exception $e) {
            // Log the exception for debugging
            \Log::error($e->getMessage());

            // Return a generic error response
            return response()->json(['error' => 'An error occurred while fetching data.'], 500);
        }
    }

    public function updateProposal(Request $request)
    {
        try {
            // Validate incoming data
            $validated = $request->validate([
                'id' => 'required|exists:manufacturing_proposals,proposal_id',  // Ensure the proposal ID exists
                'lead_id' => 'required|exists:manufacturing_leads,id',  // Ensure the lead ID exists
                'proposal_number' => 'required|string|max:255',
                'proposal_details' => 'nullable|string',
                'total_amount' => 'nullable|numeric',

            ]);

            // Log the validated request to ensure we're receiving the correct data
            \Log::info('Updating proposal with data:', $validated);

            // Find the proposal by ID
            $proposal = ManufacturingProposal::find($validated['id']);

            if (!$proposal) {
                // If proposal is not found, log the error and return a 404 response
                \Log::error('Proposal not found with ID: ' . $validated['id']);
                return response()->json(['error' => 'Proposal not found.'], 404);
            }

            // Update the proposal data
            $proposal->lead_id = $validated['lead_id'];
            $proposal->proposal_number = $validated['proposal_number'];
            $proposal->proposal_details = $validated['proposal_details'];
            $proposal->total_amount = $validated['total_amount'];
            $proposal->status = $request['status'];

            // Save the updated proposal
            $proposal->save();

            // Log success
            \Log::info('Proposal updated successfully with ID: ' . $proposal->proposal_id);

            return response()->json([
                'success' => true,
                'message' => 'Proposal updated successfully',
                'data' => $proposal
            ]);
        } catch (\Exception $e) {
            // Log full exception details for debugging
            \Log::error('Error updating proposal: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            // Return a generic error response to the client
            return response()->json(['error' => 'An error occurred while updating the proposal.'], 500);
        }
    }


    public function getProposalById($id)
    {
        try {
            // Fetch the proposal along with its related lead
            $proposal = DB::table('manufacturing_proposals')
                ->join('manufacturing_leads', 'manufacturing_proposals.lead_id', '=', 'manufacturing_leads.id')
                ->select(
                    'manufacturing_proposals.id as proposal_id',
                    'manufacturing_proposals.lead_id',
                    'manufacturing_proposals.proposal_number',
                    'manufacturing_proposals.proposal_details',
                    'manufacturing_proposals.total_amount',
                    'manufacturing_proposals.status',
                    'manufacturing_proposals.created_at',
                    'manufacturing_proposals.updated_at',
                    'manufacturing_leads.lead_name',
                    'manufacturing_leads.estimated_cost',
                    'manufacturing_leads.lead_status'
                )
                ->where('manufacturing_proposals.id', $id)
                ->first();  // Fetch only one proposal

            if (!$proposal) {
                return response()->json(['error' => 'Proposal not found.'], 404);
            }

            // Fetch all leads for the dropdown
            $leads = DB::table('manufacturing_leads')
                ->select('id', 'lead_name')
                ->get();
            $quotation_items = DB::table('quotation_items')
                ->select('*')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $proposal,
                'leads' => $leads,
                'quotation_items' => $quotation_items
            ]);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error fetching proposal: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while fetching the proposal data.'], 500);
        }
    }



    public function storeProposals(Request $request)
    {
        try {
            // Validate the incoming request data
            $validated = $request->validate([
                'lead_id' => 'required|exists:manufacturing_leads,id', // Ensure lead exists
                'proposal_number' => 'required|string',
                'proposal_details' => 'required|string',
                'total_amount' => 'required|numeric',

            ]);

            // Create a new proposal
            $proposal = ManufacturingProposal::create([
                'lead_id' => $validated['lead_id'],
                'proposal_number' => $validated['proposal_number'],
                'proposal_details' => $validated['proposal_details'],
                'total_amount' => $validated['total_amount'],
                'status' => $request['proposal_status'],
            ]);

            // Return success response
            return response()->json(['success' => true, 'message' => 'Proposal added successfully!'], 200);
        } catch (\Exception $e) {
            // Log the exception for debugging
            \Log::error($e->getMessage());

            // Return a generic error response
            return response()->json(['error' => 'An error occurred while adding the proposal.'], 500);
        }
    }


    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function create($leadID = null)
    {
        $this->addPermission = user()->permission('add_clients');

        abort_403(!in_array($this->addPermission, User::ALL_ADDED_BOTH));

        if ($leadID) {
            $this->leadDetail = Lead::findOrFail($leadID);
        }

        if (request('lead') != '') {
            $this->leadId = request('lead');
            $this->type = 'lead';
            $this->lead = Lead::findOrFail($this->leadId);
        }

        if ($this->addPermission == 'all') {
            $this->employees = User::allEmployees(null, true);
        }

        $this->pageTitle = __('app.addClient');
        $this->countries = countries();
        $this->categories = ClientCategory::all();
        $this->salutations = Salutation::cases();
        $this->languages = LanguageSetting::where('status', 'enabled')->get();

        $client = new ClientDetails();
        $getCustomFieldGroupsWithFields = $client->getCustomFieldGroupsWithFields();

        if ($getCustomFieldGroupsWithFields) {
            $this->fields = $getCustomFieldGroupsWithFields->fields;
        }

        $this->view = 'clients.ajax.create';

        if (request()->ajax()) {
            if (request('quick-form') == 1) {
                return view('clients.ajax.quick_create', $this->data);
            }

            return $this->returnAjax($this->view);
        }


        return view('clients.create', $this->data);
    }

    /**
     * XXXXXXXXXXX
     *
     * @return array
     */
    public function store(StoreClientRequest $request)
    {

        DB::beginTransaction();

        $data = $request->all();
        unset($data['country']);
        $data['password'] = bcrypt($request->password);
        $data['country_id'] = $request->country;
        $data['name'] = $request->name;
        $data['email_notifications'] = $request->sendMail == 'yes' ? 1 : 0;
        $data['gender'] = $request->gender ?? null;
        $data['locale'] = $request->locale ?? 'en';

        if ($request->has('telegram_user_id')) {
            $data['telegram_user_id'] = $request->telegram_user_id;
        }

        if ($request->hasFile('image')) {
            $data['image'] = Files::uploadLocalOrS3($request->image, 'avatar', 300);
        }

        if ($request->hasFile('company_logo')) {
            $data['company_logo'] = Files::uploadLocalOrS3($request->company_logo, 'client-logo', 300);
        }

        $user = User::create($data);
        $user->clientDetails()->create($data);
        $client_id = $user->id;

        $client_note = new ClientNote();
        $note = trim_editor($request->note);

        if ($note != '') {
            $client_note->title = 'Note';
            $client_note->client_id = $client_id;
            $client_note->details = $note;
            $client_note->save();
        }

        // To add custom fields data
        if ($request->custom_fields_data) {
            $client = $user->clientDetails;
            $client->updateCustomFieldData($request->custom_fields_data);
        }

        $role = Role::where('name', 'client')->select('id')->first();
        $user->attachRole($role->id);

        $user->assignUserRolePermission($role->id);

        // Log search
        $this->logSearchEntry($user->id, $user->name, 'clients.show', 'client');

        if (!is_null($user->email)) {
            $this->logSearchEntry($user->id, $user->email, 'clients.show', 'client');
        }

        if (!is_null($user->clientDetails->company_name)) {
            $this->logSearchEntry($user->id, $user->clientDetails->company_name, 'clients.show', 'client');
        }

        if ($request->has('lead')) {
            $lead = Lead::findOrFail($request->lead);
            $lead->client_id = $user->id;
            $lead->save();
        }

        DB::commit();

        $redirectUrl = urldecode($request->redirect_url);

        if ($redirectUrl == '') {
            $redirectUrl = route('clients.index');
        }

        if ($request->add_more == 'true') {
            $html = $this->create();

            return Reply::successWithData(__('messages.recordSaved'), ['html' => $html, 'add_more' => true]);
        }

        if ($request->has('ajax_create')) {
            $projects = Project::all();
            $teams = User::allClients();
            $options = BaseModel::options($projects, null, 'project_name');

            $teamData = '';

            foreach ($teams as $team) {
                $selected = ($team->id == $user->id) ? 'selected' : '';

                $teamData .= '<option ' . $selected . ' data-content="';

                $teamData .= '<div class=\'media align-items-center mw-250\'>';

                $teamData .= '<div class=\'position-relative\'><img src=' . $team->image_url . ' class=\'mr-2 taskEmployeeImg rounded-circle\'></div>';
                $teamData .= '<div class=\'media-body\'>';
                $teamData .= '<h5 class=\'mb-0 f-13\'>' . $team->name . '</h5>';
                $teamData .= '<p class=\'my-0 f-11 text-dark-grey\'>' . $team->email . '</p>';

                $teamData .= (!is_null($team->clientDetails->company_name)) ? '<p class=\'my-0 f-11 text-dark-grey\'>' . $team->clientDetails->company_name . '</p>' : '';
                $teamData .= '</div>';
                $teamData .= '</div>"';

                $teamData .= 'value="' . $team->id . '"> ' . $team->name . '';

                $teamData .= '</option>';
            }

            return Reply::successWithData(__('messages.recordSaved'), ['teamData' => $teamData, 'project' => $options, 'redirectUrl' => $redirectUrl]);
        }

        return Reply::successWithData(__('messages.recordSaved'), ['redirectUrl' => $redirectUrl]);
    }

    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $this->client = User::withoutGlobalScope(ActiveScope::class)->with('clientDetails')->findOrFail($id);
        $this->editPermission = user()->permission('edit_clients');

        abort_403(!($this->editPermission == 'all' || ($this->editPermission == 'added' && $this->client->clientDetails->added_by == user()->id) || ($this->editPermission == 'both' && $this->client->clientDetails->added_by == user()->id)));

        $this->countries = countries();
        $this->categories = ClientCategory::all();

        if ($this->editPermission == 'all') {
            $this->employees = User::allEmployees();
        }

        $this->pageTitle = __('app.update') . ' ' . __('app.client');
        $this->salutations = Salutation::cases();
        $this->languages = LanguageSetting::where('status', 'enabled')->get();

        if (!is_null($this->client->clientDetails)) {
            $this->clientDetail = $this->client->clientDetails->withCustomFields();

            $getCustomFieldGroupsWithFields = $this->clientDetail->getCustomFieldGroupsWithFields();

            if ($getCustomFieldGroupsWithFields) {
                $this->fields = $getCustomFieldGroupsWithFields->fields;
            }
        }

        $this->subcategories = ClientSubCategory::all();

        $this->view = 'clients.ajax.edit';

        if (request()->ajax()) {
            return $this->returnAjax($this->view);
        }

        return view('clients.create', $this->data);
    }

    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateClientRequest $request, $id)
    {
        $user = User::withoutGlobalScope(ActiveScope::class)->findOrFail($id);
        $data = $request->all();


        unset($data['password']);
        unset($data['country']);

        if ($request->password != '') {
            $data['password'] = bcrypt($request->password);
        }

        $data['country_id'] = $request->country;

        if ($request->has('sendMail')) {
            $user->email_notifications = $request->sendMail == 'yes' ? 1 : 0;
        }

        if ($request->has('telegram_user_id')) {
            $data['telegram_user_id'] = $request->telegram_user_id;
        }


        if ($request->image_delete == 'yes') {
            Files::deleteFile($user->image, 'avatar');
            $data['image'] = null;
        }

        if ($request->hasFile('image')) {

            Files::deleteFile($user->image, 'avatar');
            $data['image'] = Files::uploadLocalOrS3($request->image, 'avatar', 300);
        }


        $user->update($data);

        if ($user->clientDetails) {
            $data['category_id'] = $request->category_id;
            $data['sub_category_id'] = $request->sub_category_id;
            $data['note'] = trim_editor($request->note);
            $data['locale'] = $request->locale;
            $fields = $request->only($user->clientDetails->getFillable());

            if ($request->has('company_logo_delete') && $request->company_logo_delete == 'yes') {
                Files::deleteFile($user->clientDetails->company_logo, 'client-logo');
                $fields['company_logo'] = null;
            }

            if ($request->hasFile('company_logo')) {
                Files::deleteFile($user->clientDetails->company_logo, 'client-logo');
                $fields['company_logo'] = Files::uploadLocalOrS3($request->company_logo, 'client-logo', 300);
            }

            $user->clientDetails->fill($fields);
            $user->clientDetails->save();
        } else {
            $user->clientDetails()->create($data);
        }

        // To add custom fields data
        if ($request->custom_fields_data) {
            $user->clientDetails->updateCustomFieldData($request->custom_fields_data);
        }

        $this->createEmployeeActivity(auth()->user()->id, 'client-updated', $id, 'client');


        $redirectUrl = urldecode($request->redirect_url);

        if ($redirectUrl == '') {
            $redirectUrl = route('clients.index');
        }

        return Reply::successWithData(__('messages.updateSuccess'), ['redirectUrl' => $redirectUrl]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $this->client = User::withoutGlobalScope(ActiveScope::class)->with('clientDetails')->findOrFail($id);
        $this->deletePermission = user()->permission('delete_clients');

        abort_403(
            !($this->deletePermission == 'all'
                || ($this->deletePermission == 'added' && $this->client->clientDetails->added_by == user()->id)
                || ($this->deletePermission == 'both' && $this->client->clientDetails->added_by == user()->id)
            )
        );

        $this->deleteClient($this->client);

        return Reply::success(__('messages.deleteSuccess'));
    }

    private function deleteClient(User $user)
    {
        $universalSearches = UniversalSearch::where('searchable_id', $user->id)->where('module_type', 'client')->get();

        if ($universalSearches) {
            foreach ($universalSearches as $universalSearch) {
                UniversalSearch::destroy($universalSearch->id);
            }
        }


        Notification::whereNull('read_at')
            ->where(function ($q) use ($user) {
                $q->where('data', 'like', '{"id":' . $user->id . ',%');
                $q->orWhere('data', 'like', '%,"name":' . $user->name . ',%');
                $q->orWhere('data', 'like', '%,"user_one":' . $user->id . ',%');
                $q->orWhere('data', 'like', '%,"client_id":' . $user->id . '%');
            })->delete();

        $user->delete();

        Lead::where('client_id', $user->id)->update(['client_id' => null]);
        $this->createEmployeeActivity(auth()->user()->id, 'client-deleted');
    }

    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function applyQuickAction(Request $request)
    {
        switch ($request->action_type) {
            case 'delete':
                $this->deleteRecords($request);

                return Reply::success(__('messages.deleteSuccess'));
            case 'change-status':
                $this->changeStatus($request);

                return Reply::success(__('messages.updateSuccess'));
            default:
                return Reply::error(__('messages.selectAction'));
        }
    }

    protected function deleteRecords($request)
    {
        abort_403(user()->permission('delete_clients') !== 'all');
        $users = User::withoutGlobalScope(ActiveScope::class)->whereIn('id', explode(',', $request->row_ids))->get();
        $users->each(function ($user) {
            $this->deleteClient($user);
        });

        return true;
    }

    protected function changeStatus($request)
    {
        abort_403(user()->permission('edit_clients') !== 'all');
        User::withoutGlobalScope(ActiveScope::class)
            ->whereIn('id', explode(',', $request->row_ids))
            ->update(['status' => $request->status]);

        return true;
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $this->client = User::withoutGlobalScope(ActiveScope::class)->findOrFail($id);
        $this->clientLanguage = LanguageSetting::where('language_code', $this->client->locale)->first();
        $this->viewPermission = user()->permission('view_clients');
        $this->viewDocumentPermission = user()->permission('view_client_document');

        if (!$this->client->hasRole('client')) {
            abort(404);
        }

        abort_403(!($this->viewPermission == 'all'
            || ($this->viewPermission == 'added' && $this->client->clientDetails->added_by == user()->id)
            || ($this->viewPermission == 'both' && $this->client->clientDetails->added_by == user()->id)));

        $this->pageTitle = $this->client->name;

        $this->clientStats = $this->clientStats($id);
        $this->projectChart = $this->projectChartData($id);
        $this->invoiceChart = $this->invoiceChartData($id);

        $this->earningTotal = Payment::leftJoin('invoices', 'invoices.id', '=', 'payments.invoice_id')
            ->leftJoin('projects', 'projects.id', '=', 'payments.project_id')
            ->where(function ($q) use ($id) {
                $q->where('invoices.client_id', $id);
                $q->orWhere('projects.client_id', $id);
            })->sum('amount');

        $this->view = 'clients.ajax.profile';

        $tab = request('tab');

        switch ($tab) {
            case 'projects':
                return $this->projects();
            case 'invoices':
                return $this->invoices();
            case 'payments':
                return $this->payments();
            case 'estimates':
                return $this->estimates();
            case 'creditnotes':
                return $this->creditnotes();
            case 'contacts':
                return $this->contacts();
            case 'orders':
                return $this->orders();
            case 'documents':
                abort_403(!($this->viewDocumentPermission == 'all'
                    || ($this->viewDocumentPermission == 'added' && $this->client->clientDetails->added_by == user()->id)
                    || ($this->viewDocumentPermission == 'owned' && $this->client->clientDetails->user_id == user()->id)
                    || ($this->viewDocumentPermission == 'both' && ($this->client->clientDetails->added_by == user()->id || $this->client->clientDetails->user_id == user()->id))));

                $this->view = 'clients.ajax.documents';
                break;
            case 'notes':
                return $this->notes();
            case 'tickets':
                return $this->tickets();
            case 'gdpr':
                $this->client = User::withoutGlobalScope(ActiveScope::class)->findOrFail($id);
                $this->consents = PurposeConsent::with(['user' => function ($query) use ($id) {
                    $query->where('client_id', $id)
                        ->orderByDesc('created_at');
                }])->get();

                return $this->gdpr();
            default:
                $this->clientDetail = ClientDetails::where('user_id', '=', $this->client->id)->first();

                if (!is_null($this->clientDetail)) {
                    $this->clientDetail = $this->clientDetail->withCustomFields();

                    $getCustomFieldGroupsWithFields = $this->clientDetail->getCustomFieldGroupsWithFields();

                    if ($getCustomFieldGroupsWithFields) {
                        $this->fields = $getCustomFieldGroupsWithFields->fields;
                    }
                }

                $this->view = 'clients.ajax.profile';
                break;
        }

        if (request()->ajax()) {
            return $this->returnAjax($this->view);
        }

        $this->activeTab = $tab ?: 'profile';

        return view('clients.show', $this->data);
    }

    public function clientStats($id)
    {
        return DB::table('users')
            ->select(
                DB::raw('(select count(projects.id) from `projects` WHERE projects.client_id = ' . $id . ' and deleted_at IS NULL) as totalProjects'),
                DB::raw('(select count(invoices.id) from `invoices` left join projects on projects.id=invoices.project_id WHERE invoices.status != "paid" and invoices.status != "canceled" and (projects.client_id = ' . $id . ' or invoices.client_id = ' . $id . ')) as totalUnpaidInvoices'),
                DB::raw('(select sum(payments.amount) from `payments` left join projects on projects.id=payments.project_id WHERE payments.status = "complete" and projects.client_id = ' . $id . ') as projectPayments'),
                DB::raw('(select sum(payments.amount) from `payments` inner join invoices on invoices.id=payments.invoice_id  WHERE payments.status = "complete" and invoices.client_id = ' . $id . ') as invoicePayments'),
                DB::raw('(select count(contracts.id) from `contracts` WHERE contracts.client_id = ' . $id . ') as totalContracts')
            )
            ->first();
    }

    /**
     * XXXXXXXXXXX
     *
     * @return array
     */
    public function projectChartData($id)
    {
        // Fetch active project status settings
        $statusSettings = ProjectStatusSetting::where('status', 'active')->get();

        // Initialize data array
        $data = [
            'labels' => $statusSettings->pluck('status_name'),
            'colors' => $statusSettings->pluck('color'),
            'values' => []
        ];

        // Construct the query to count projects for each status
        $query = Project::selectRaw('COUNT(projects.id) as count, pss.status_name as label')
            ->join('project_status_settings as pss', 'projects.status', '=', 'pss.status')
            ->where('pss.status', 'active')
            ->where('projects.client_id', $id)
            ->groupBy('pss.status_name');

        // Execute the query and fetch counts for each status
        $statusCounts = $query->pluck('count', 'label');

        // Populate the values array with counts for each status
        foreach ($data['labels'] as $label) {
            $data['values'][] = $statusCounts[$label] ?? 0;
        }

        return $data;
    }

    /**
     * XXXXXXXXXXX
     *
     * @return array
     */
    public function invoiceChartData($id)
    {
        // Define labels, translations, and colors
        $labels = ['paid', 'unpaid', 'partial', 'canceled', 'draft'];
        $translations = [__('app.paid'), __('app.unpaid'), __('app.partial'), __('app.canceled'), __('app.draft')];
        $colors = ['#2CB100', '#FCBD01', '#1d82f5', '#D30000', '#616e80'];

        // Construct the query to count invoices for each status
        $query = Invoice::selectRaw('COUNT(*) as count, status')
            ->where('client_id', $id)
            ->whereIn('status', $labels)
            ->groupBy('status');

        // Execute the query and fetch counts for each status
        $statusCounts = $query->pluck('count', 'status');

        // Initialize data array
        $data = [
            'labels' => $translations,
            'colors' => $colors,
            'values' => []
        ];

        // Populate the values array with counts for each status
        foreach ($labels as $label) {
            $data['values'][] = $statusCounts[$label] ?? 0;
        }

        return $data;
    }

    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function projectList($id)
    {
        if ($id != 0) {
            $projects = Project::where('client_id', $id)->get();
            $options = BaseModel::options($projects, null, 'project_name');
        } else {
            $options = '<option value="">--</option>';
        }

        return Reply::dataOnly(['status' => 'success', 'data' => $options]);
    }

    /**
     * XXXXXXXXXXX
     *
     * @return \Illuminate\Http\Response
     */
    public function ajaxDetails($id)
    {
        if ($id != 0) {
            $client = User::withoutGlobalScope(ActiveScope::class)->with('clientDetails', 'country')->find($id);
        } else {
            $client = null;
        }

        $clientProjects = Project::where('client_id', $id)->get();

        $options = '<option value="">--</option>';

        foreach ($clientProjects as $project) {

            $options .= '<option value="' . $project->id . '"> ' . $project->project_name . ' </option>';
        }

        $data = $client ?: null;

        return Reply::dataOnly(['status' => 'success', 'data' => $data, 'project' => $options]);
    }

    public function projects()
    {

        $viewPermission = user()->permission('view_projects');

        abort_403(!($viewPermission == 'all' || $viewPermission == 'added'));
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.projects';

        $dataTable = new ProjectsDataTable();

        return $dataTable->render('clients.show', $this->data);
    }

    public function invoices()
    {
        $dataTable = new InvoicesDataTable();
        $viewPermission = user()->permission('view_invoices');

        abort_403(!in_array($viewPermission, ['all', 'added', 'owned', 'both']));
        $tab = request('tab');

        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.invoices';

        return $dataTable->render('clients.show', $this->data);
    }

    public function payments()
    {
        $dataTable = new PaymentsDataTable();
        $viewPermission = user()->permission('view_payments');

        abort_403(!($viewPermission == 'all' || $viewPermission == 'added'));
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.payments';

        return $dataTable->render('clients.show', $this->data);
    }

    public function estimates()
    {
        $dataTable = new EstimatesDataTable();
        $viewPermission = user()->permission('view_estimates');

        abort_403(!in_array($viewPermission, ['all', 'added', 'owned', 'both']));

        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';
        $this->view = 'clients.ajax.estimates';

        return $dataTable->render('clients.show', $this->data);
    }

    public function creditnotes()
    {
        $dataTable = new CreditNotesDataTable();
        $viewPermission = user()->permission('view_invoices');

        abort_403($viewPermission == 'none');
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';
        $this->view = 'clients.ajax.credit_notes';

        return $dataTable->render('clients.show', $this->data);
    }

    public function contacts()
    {
        $dataTable = new ClientContactsDataTable();
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.contacts';

        return $dataTable->render('clients.show', $this->data);
    }

    public function notes()
    {
        $dataTable = new ClientNotesDataTable();
        $viewPermission = user()->permission('view_client_note');

        abort_403(($viewPermission == 'none'));
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';
        $this->view = 'clients.ajax.notes';

        return $dataTable->render('clients.show', $this->data);
    }

    public function tickets()
    {
        $dataTable = new TicketDataTable();
        $viewPermission = user()->permission('view_clients');

        abort_403(!($viewPermission == 'all' || $viewPermission == 'added' || $viewPermission == 'both'));
        $tab = request('tab');
        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.tickets';

        return $dataTable->render('clients.show', $this->data);
    }

    public function gdpr()
    {
        $dataTable = new ClientGDPRDataTable();
        $tab = request('tab');
        $this->activeTab = $tab ?: 'gdpr';

        $this->view = 'clients.ajax.gdpr';

        return $dataTable->render('clients.show', $this->data);
    }

    public function consent(Request $request)
    {
        $clientId = $request->clientId;
        $this->consentId = $request->consentId;
        $this->clientId = $clientId;

        $this->consent = PurposeConsent::with(['user' => function ($query) use ($request) {
            $query->where('client_id', $request->clientId)
                ->orderByDesc('created_at');
        }])
            ->where('id', $request->consentId)
            ->first();

        return view('clients.gdpr.consent-form', $this->data);
    }

    public function saveClientConsent(SaveConsentUserDataRequest $request, $id)
    {
        $user = User::findOrFail($id);
        $consent = PurposeConsent::findOrFail($request->consent_id);

        if ($request->consent_description && $request->consent_description != '') {
            $consent->description = trim_editor($request->consent_description);
            $consent->save();
        }

        // Saving Consent Data
        $newConsentLead = new PurposeConsentUser();
        $newConsentLead->client_id = $user->id;
        $newConsentLead->purpose_consent_id = $consent->id;
        $newConsentLead->status = trim($request->status);
        $newConsentLead->ip = $request->ip();
        $newConsentLead->updated_by_id = $this->user->id;
        $newConsentLead->additional_description = $request->additional_description;
        $newConsentLead->save();

        return $request->status == 'agree' ? Reply::success(__('messages.consentOptIn')) : Reply::success(__('messages.consentOptOut'));
    }

    public function approve($id)
    {
        abort_403(!in_array('admin', user_roles()));

        User::where('id', $id)->update(
            ['admin_approval' => 1]
        );

        $userSession = new AppSettingController();
        $userSession->deleteSessions([$id]);

        return Reply::success(__('messages.updateSuccess'));
    }

    public function importClient()
    {
        $this->pageTitle = __('app.importExcel') . ' ' . __('app.client');

        $addPermission = user()->permission('add_clients');
        abort_403(!in_array($addPermission, ['all', 'added', 'both']));

        $this->view = 'clients.ajax.import';

        if (request()->ajax()) {
            return $this->returnAjax($this->view);
        }

        return view('clients.create', $this->data);
    }

    public function importStore(ImportRequest $request)
    {
        $this->importFileProcess($request, ClientImport::class);

        $view = view('clients.ajax.import_progress', $this->data)->render();

        return Reply::successWithData(__('messages.importUploadSuccess'), ['view' => $view]);
    }

    public function importProcess(ImportProcessRequest $request)
    {
        $batch = $this->importJobProcess($request, ClientImport::class, ImportClientJob::class);

        return Reply::successWithData(__('messages.importProcessStart'), ['batch' => $batch]);
    }

    public function financeCount(Request $request)
    {
        $id = $request->id;

        $counts = User::withCount('projects', 'invoices', 'estimates')->withoutGlobalScope(ActiveScope::class)->find($id);

        $payments = Payment::leftJoin('invoices', 'invoices.id', '=', 'payments.invoice_id')
            ->leftJoin('projects', 'projects.id', '=', 'payments.project_id')
            ->leftJoin('orders', 'orders.id', '=', 'payments.order_id')
            ->where(function ($query) use ($id) {
                $query->where('projects.client_id', $id)
                    ->orWhere('invoices.client_id', $id)
                    ->orWhere('orders.client_id', $id);
            })->count();

        $projectName = $counts->projects_count > 1 ? __('app.menu.projects') : __('app.project');
        $invoiceName = $counts->invoices_count > 1 ? __('app.menu.invoices') : __('app.invoice');
        $estimateName = $counts->estimates_count > 1 ? __('app.menu.estimates') : __('app.estimate');
        $paymentName = $payments > 1 ? __('app.menu.payments') : __('app.payment');

        $deleteClient = (__('messages.clientFinanceCount', ['projectCount' => $counts->projects_count, 'invoiceCount' => $counts->invoices_count, 'estimateCount' => $counts->estimates_count, 'paymentCount' => $payments, 'project' => $projectName, 'invoice' => $invoiceName, 'estimate' => $estimateName, 'payment' => $paymentName]));

        return Reply::dataOnly(['status' => 'success', 'deleteClient' => $deleteClient]);
    }

    public function clientDetails(Request $request)
    {
        $teamData = '';

        if ($request->id == 0) {
            $clients = User::allClients();

            foreach ($clients as $client) {

                $teamData .= '<option data-content="';

                $teamData .= '<div class=\'media align-items-center mw-250\'>';

                $teamData .= '<div class=\'position-relative\'><img src=' . $client->image_url . ' class=\'mr-2 taskEmployeeImg rounded-circle\'></div>';
                $teamData .= '<div class=\'media-body\'>';
                $teamData .= '<h5 class=\'mb-0 f-13\'>' . $client->name . '</h5>';
                $teamData .= '<p class=\'my-0 f-11 text-dark-grey\'>' . $client->email . '</p>';

                $teamData .= (!is_null($client->clientDetails->company_name)) ? '<p class=\'my-0 f-11 text-dark-grey\'>' . $client->clientDetails->company_name . '</p>' : '';
                $teamData .= '</div>';
                $teamData .= '</div>"';

                $teamData .= 'value="' . $client->id . '"> ' . $client->name . '';

                $teamData .= '</option>';
            }
        } else {

            $project = Project::with('client')->findOrFail($request->id);

            if ($project->client != null) {

                $teamData .= '<option data-content="';

                $teamData .= '<div class=\'media align-items-center mw-250\'>';

                $teamData .= '<div class=\'position-relative\'><img src=' . $project->client->image_url . ' class=\'mr-2 taskEmployeeImg rounded-circle\'></div>';
                $teamData .= '<div class=\'media-body\'>';
                $teamData .= '<h5 class=\'mb-0 f-13\'>' . $project->client->name . '</h5>';
                $teamData .= '<p class=\'my-0 f-11 text-dark-grey\'>' . $project->client->email . '</p>';

                $teamData .= (!is_null($project->client->company->company_name)) ? '<p class=\'my-0 f-11 text-dark-grey\'>' . $project->client->company->company_name . '</p>' : '';
                $teamData .= '</div>';
                $teamData .= '</div>"';

                $teamData .= 'value="' . $project->client->id . '"> ' . $project->client->name . '';

                $teamData .= '</option>';
            }
        }

        return Reply::dataOnly(['teamData' => $teamData]);
    }

    public function orders()
    {
        $dataTable = new OrdersDataTable();
        $viewPermission = user()->permission('view_order');
        abort_403(!in_array($viewPermission, ['all', 'added', 'owned', 'both']));

        $tab = request('tab');

        $this->activeTab = $tab ?: 'profile';

        $this->view = 'clients.ajax.orders';

        return $dataTable->render('clients.show', $this->data);
    }


    // Method to handle proposal conversion
    public function convertProposalToProject(Request $request)
    {
        $validated = $request->validate([
            'country' => 'required|string|max:255',
            'medium' => 'required|string|max:50',
            'freight_terms' => 'required|string|max:50',
            'delivery_address' => 'required|string',
            'shipping_instructions' => 'nullable|string',
            'contact_person' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:50',
            'payment_terms' => 'nullable|string|max:50',
            'customs_clearance_required' => 'required|boolean',
            'customs_documentation' => 'nullable|string',
        ]);

        // Fetch the proposal using the provided project_id from the request
        $proposal = ManufacturingProposal::find($request->input('project_id'));

        // If proposal is not found, return error response
        if (!$proposal) {
            return response()->json(['status' => 'error', 'message' => 'Proposal not found'], 404);
        }

        try {
            DB::beginTransaction(); // Start a transaction to ensure data integrity

            // Create the project from the proposal
            $project = new ManufacturingProject();
            $project->order_id = "3";  // Assuming order_id exists in proposal, or change as needed
            $project->lead_id = $proposal->lead_id;
            $project->project_name = 'Project for ' . $proposal->proposal_number;
            $project->status = 'Not Started'; // Default status
            $project->roadmap = json_encode([]); // Placeholder empty JSON for roadmap
            $project->save(); // Save the new project

            // Retrieve the newly generated project id
            $projectId = $project->id;

            // Update the proposal status to indicate that it has been converted to a project
            $proposal->proposal_status = 1;  // Assuming '1' means "Converted"
            $proposal->save();

            // Save the dispatch info to the database
            $dispatchInfo = new ClientDispatchInfo();
            $dispatchInfo->project_id = $projectId; // Use the project id of the newly created project
            $dispatchInfo->country = $validated['country'];
            $dispatchInfo->medium = $validated['medium'];
            $dispatchInfo->freight_terms = $validated['freight_terms'];
            $dispatchInfo->delivery_address = $validated['delivery_address'];
            $dispatchInfo->shipping_instructions = $validated['shipping_instructions'];
            $dispatchInfo->contact_person = $validated['contact_person'];
            $dispatchInfo->contact_email = $validated['contact_email'];
            $dispatchInfo->contact_phone = $validated['contact_phone'];
            $dispatchInfo->payment_terms = $validated['payment_terms'];
            $dispatchInfo->customs_clearance_required = $validated['customs_clearance_required'];
            $dispatchInfo->customs_documentation = $validated['customs_documentation'];
            $dispatchInfo->save(); // Save the dispatch info

            DB::commit(); // Commit the transaction if everything is successful

            // Return a success response with the project id
            return response()->json(['status' => 'success', 'message' => 'Proposal converted to project successfully!', 'project_id' => $projectId]);
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback if any error occurs
            Log::error('Error converting proposal to project: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'Error occurred while converting the proposal to a project. Please try again later.'], 500);
        }
    }


    public function getProjects(Request $request)
    {
        // Fetch projects with a join to the users table to get the client name and the client_details table to get company name
        $projects = Project::select([
            'projects.id',
            'projects.project_name',
            'projects.project_short_code',
            'projects.start_date',
            'projects.deadline',
            'projects.status',
            'projects.allocation_status',
            'projects.client_id',  // Include client_id to join with the users table
            'users.name as client_name', // Join users table and get client name
            'users.id as user_id', // Join users table and get client name
            'client_details.company_name', // Join client_details table and get company name
        ])
            ->leftJoin('users', 'projects.client_id', '=', 'users.id') // Left join with users table for client name
            ->leftJoin('client_details', 'users.id', '=', 'client_details.user_id') // Left join client_details table to get company name
            ->orderBy('projects.created_at', 'desc')  // Optionally order the projects by created_at
            ->paginate($request->input('length', 10));  // Paginate based on the 'length' parameter from DataTables

        // Format the data as needed for the DataTable response
        $projectsData = $projects->map(function ($project) {
            return [
                'id' => $project->id,
                'project_name' => $project->project_name,
                'project_short_code' => $project->project_short_code,
                'start_date' => $project->start_date,
                'deadline' => $project->deadline,
                'status' => $project->status,
                'user_id' => $project->user_id,
                'allocation_status' => $project->allocation_status,
                'client_name' => $project->client_name ?: 'No Client',  // Access the client_name from the join
                'company_name' => $project->company_name ?: 'No Company',  // Access company_name from client_details join
                'members' => [], // Add members field if required (implement the logic if needed)
            ];
        });

        // Return the formatted response with necessary pagination data
        return response()->json([
            'draw' => $request->input('draw'),
            'recordsTotal' => $projects->total(),  // Total records without filtering
            'recordsFiltered' => $projects->total(),  // Total records after filtering (can adjust if you add search functionality)
            'data' => $projectsData,  // Paginated and formatted data
        ]);
    }


    // Method to fetch inventory materials
    public function getInventoryMaterials()
    {
        $materials = ManufacturingInventory::select('id', 'material_name')->get();
        return response()->json(['materials' => $materials]);
    }

    // Method to fetch the details of a specific inventory item
    public function getInventoryDetails($id)
    {
        $inventory = ManufacturingInventory::find($id);

        if ($inventory) {
            return response()->json([
                'stock_quantity' => $inventory->quantity_in_stock,
                'location' => $inventory->location,
                'unit_price' => $inventory->unit_price
            ]);
        }

        return response()->json(['message' => 'Material not found'], 404);
    }



    public function allocateInventory(Request $request)
    {
        // Start a transaction to ensure atomicity
        DB::beginTransaction();

        try {
            // Find the project and inventory by their IDs
            $project = Project::find($request->project_id);
            $inventory = ManufacturingInventory::find($request->material_name);

            // Check if there is enough stock
            if ($inventory->quantity_in_stock < $request->project_quantity) {
                return response()->json(['message' => 'Not enough stock available. Please purchase more.'], 400);
            }

            // Create the allocation
            $allocation = InventoryAllocation::create([
                'project_id' => $project->id,
                'inventory_id' => $inventory->id,
                'quantity' => $request->project_quantity,
                'location' => $request->location,
                'unit_price' => $request->unit_price,
                'reorder_level' => "3"
            ]);

            // Deduct the allocated stock from the inventory
            $inventory->quantity_in_stock -= $request->project_quantity;
            $inventory->save();

            // Update project allocation status based on the allocation
            $project->allocation_status = 'Waiting for Inventory Store Approval'; // After allocation
            $project->save();

            // Commit the transaction
            DB::commit();

            return response()->json(['message' => 'Inventory successfully allocated!', 'allocation' => $allocation]);
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollback();

            // Log the error for debugging
            \Log::error('Error in allocating inventory: ' . $e->getMessage());

            return response()->json(['message' => 'Failed to allocate inventory. Please try again.'], 500);
        }
    }


    public function getPendingAllocations()
    {
        try {
            // Join manufacturing_projects with inventory_allocations based on project_id
            // Join inventory_allocations with inventory (or materials) table based on material_id
            $pendingAllocations = InventoryAllocation::join('projects', 'inventory_allocations.project_id', '=', 'projects.id')
                ->join('inventory', 'inventory_allocations.inventory_id', '=', 'inventory.id') // Adjust according to your table/column names
                ->where('projects.allocation_status', 'Waiting for Inventory Store Approval')
                ->select(
                    'projects.project_name',     // From manufacturing_projects
                    'inventory.material_name',                 // From inventory (or materials)
                    'inventory.location',                      // From inventory
                    'inventory.unit_price',                    // From inventory
                    'inventory.material_code',                 // From inventory
                    'inventory_allocations.quantity',          // From inventory_allocations
                    'projects.allocation_status', // From inventory_allocations
                    'inventory_allocations.id as allocation_id' // Include allocation_id for approval actions
                )
                ->get();

            // If no pending allocations, return a custom message
            // Check if there are any pending allocations
            if ($pendingAllocations->isEmpty()) {
                return response()->json([
                    'draw' => 1,  // Same as the "draw" from the DataTable request
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []  // Empty data when no allocations are found
                ]);
            }

            return response()->json([
                'draw' => 1,
                'recordsTotal' => $pendingAllocations->count(),
                'recordsFiltered' => $pendingAllocations->count(),
                'data' => $pendingAllocations
            ]);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error fetching pending allocations: ' . $e->getMessage());

            // Return a server error response with the error message
            return response()->json(['message' => 'Server Error', 'error' => $e->getMessage()], 500);
        }
    }

    public function approveAllocation($allocationId)
    {
        try {
            // Find the allocation record by ID
            $allocation = InventoryAllocation::findOrFail($allocationId);

            // Optionally, update the status of the related project
            $project = ManufacturingProject::findOrFail($allocation->project_id);
            $project->allocation_status = 'Allocated';  // or your desired status
            $project->save();

            return response()->json(['message' => 'Allocation approved successfully.']);
        } catch (\Exception $e) {
            // Log the error for debugging purposes
            \Log::error('Error approving allocation: ' . $e->getMessage());

            // Return a server error response
            return response()->json(['message' => 'Failed to approve allocation. Please try again.'], 500);
        }
    }



    public function allocateMaterialToProject(Request $request)
    {
        $projectId = $request->input('project_id');
        $materialId = $request->input('material_id');
        $quantity = $request->input('quantity');

        // Validate inputs
        if (!$projectId || !$materialId || !$quantity) {
            return response()->json(['status' => 'error', 'message' => 'Missing required parameters'], 400);
        }

        // Find the project and material
        $project = ManufacturingProject::find($projectId);
        $material = ManufacturingInventory::find($materialId);

        if (!$project || !$material) {
            return response()->json(['status' => 'error', 'message' => 'Project or material not found'], 404);
        }

        // Check if enough material is available in stock
        if ($material->quantity_in_stock < $quantity) {
            return response()->json(['status' => 'error', 'message' => 'Insufficient stock'], 400);
        }

        // Create ProjectMaterial pivot entry
        $projectMaterial = new ProjectMaterial();
        $projectMaterial->project_id = $project->id;
        $projectMaterial->material_id = $material->id;
        $projectMaterial->quantity = $quantity;
        $projectMaterial->save();

        // Update the inventory
        $material->quantity_in_stock -= $quantity;
        $material->save();

        return response()->json(['status' => 'success', 'message' => 'Material allocated successfully']);
    }


    public function fetchInventory()
    {
        // Fetch all inventory items from the database
        $inventoryItems = ManufacturingInventory::all();

        // Return the data as a JSON response
        return response()->json($inventoryItems);
    }


    public function checkInventory(Request $request)
    {
        $product_id = $request->input('product_id');
        $required_quantity = $request->input('quantity');

        // Get the product from the inventory
        $inventory = ManufacturingInventory::find($product_id);

        if ($inventory->isStockAvailable($required_quantity)) {
            return response()->json(['status' => 'success', 'message' => 'Stock available', 'inventory' => $inventory]);
        } else {
            return response()->json(['status' => 'error', 'message' => 'Stock unavailable'], 400);
        }
    }

    // Handle Raw Material Purchase (Vendor Purchase Request)
    public function purchaseRawMaterial(Request $request)
    {
        $productId = $request->input('product_id');
        $quantity = $request->input('quantity');

        $product = ManufacturingInventory::find($productId);

        if (!$product) {
            return response()->json(['message' => 'Product not found.'], 404);
        }

        $product->stock_quantity += $quantity;
        $product->save();

        return response()->json(['message' => 'Stock purchased successfully.']);
    }
    // Handle Raw Material Allocation
    public function allocateRawMaterial(Request $request)
    {
        $productId = $request->input('product_id');
        $quantity = $request->input('quantity');

        $product = ManufacturingInventory::find($productId);

        if (!$product) {
            return response()->json(['message' => 'Product not found.'], 404);
        }

        if ($product->stock_quantity >= $quantity) {
            $product->stock_quantity -= $quantity;
            $product->save();

            return response()->json(['message' => 'Stock allocated successfully.']);
        }

        return response()->json(['message' => 'Insufficient stock to allocate.'], 400);
    }

    // Approve Purchase Request
    public function approvePurchaseRequest($id)
    {
        $purchaseRequest = PurchaseRequest::find($id);

        if ($purchaseRequest) {
            $purchaseRequest->status = 'approved';
            $purchaseRequest->approved_by = Auth::id();
            $purchaseRequest->save();

            // Optionally, you could trigger the purchase order with the vendor here
            return response()->json(['status' => 'success', 'message' => 'Purchase request approved']);
        }

        return response()->json(['status' => 'error', 'message' => 'Purchase request not found'], 404);
    }

    // Approve Raw Material Allocation
    public function approveRawMaterialAllocation($id)
    {
        $allocation = Allocation::find($id);

        if ($allocation) {
            // Implement any logic for approval here, like notifying the lab or changing status
            return response()->json(['status' => 'success', 'message' => 'Allocation approved']);
        }

        return response()->json(['status' => 'error', 'message' => 'Allocation not found'], 404);
    }

    public function getProjectDetails(Request $request)
    {
        // Start the query for manufacturing projects
        $projects = ManufacturingProject::query();

        // Optionally apply filters if any are passed through AJAX (e.g., status filter)
        if ($request->has('status')) {
            $projects->where('allocation_status', "Allocated");
        }

        // Join with the necessary tables
        $projects = $projects->leftJoin('inventory_allocations', 'manufacturing_projects.id', '=', 'inventory_allocations.project_id')
            ->leftJoin('synthesis_stages', 'manufacturing_projects.id', '=', 'synthesis_stages.project_id') // Join synthesis stages
            ->leftJoin('stage_comments', 'synthesis_stages.id', '=', 'stage_comments.stage_id') // Join stage comments
            ->leftJoin('manager_approval', 'synthesis_stages.id', '=', 'manager_approval.stage_id') // Join manager approvals
            ->select(
                'manufacturing_projects.*', // Select all fields from ManufacturingProject
                DB::raw('SUM(inventory_allocations.quantity) as total_allocated'), // Sum of quantities allocated
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.stage_name ORDER BY synthesis_stages.id ASC) as stage_names'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.batch_number ORDER BY synthesis_stages.id ASC) as batch_numbers'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.progress ORDER BY synthesis_stages.id ASC) as progresses'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.id ORDER BY synthesis_stages.id ASC) as stage_ids'), // Get the IDs of the stages
                DB::raw('GROUP_CONCAT(DISTINCT stage_comments.comment ORDER BY synthesis_stages.id ASC) as stage_comments'),
                DB::raw('GROUP_CONCAT(DISTINCT manager_approval.manager_approval_status ORDER BY synthesis_stages.id ASC) as approval_statuses'),
                DB::raw('GROUP_CONCAT(DISTINCT manager_approval.comments ORDER BY synthesis_stages.id ASC) as approval_comments')
            )
            ->where('manufacturing_projects.allocation_status', "Allocated") // Filter for allocated projects
            ->groupBy('manufacturing_projects.id') // Group by project_id, ensuring only one record per project
            ->paginate(10);

        // Convert the items to a collection and map over it
        $projectsData = collect($projects->items())->map(function ($project) {
            $stages = explode(',', $project->stage_names);
            $progresses = explode(',', $project->progresses);
            $stageIds = explode(',', $project->stage_ids); // Get stage IDs
            $approvalStatuses = explode(',', $project->approval_statuses); // Get approval statuses
            $t3StageIndex = array_search('T3', $stages);
            $t3Progress = $t3StageIndex !== false ? $progresses[$t3StageIndex] : null;

            // If the project has the T3 stage, get the corresponding stage ID
            $t3StageId = $t3StageIndex !== false ? $stageIds[$t3StageIndex] : null;

            // Check if the project is waiting for manager approval
            $waitingForApproval = in_array(2, $approvalStatuses); // Check if any stage has manager_approval_status 2

            // Add the T3 stage logic and waiting for approval status to the project data
            $project->is_t3 = $t3StageIndex !== false;
            $project->t3_progress = $t3Progress;
            $project->t3_stage_id = $t3StageId; // Add the T3 stage ID
            $project->waiting_for_approval = $waitingForApproval; // Add waiting for approval status

            return $project;
        });

        return response()->json([
            'draw' => $request->input('draw'),
            'recordsTotal' => $projects->total(),
            'recordsFiltered' => $projects->count(),
            'data' => $projectsData,
        ]);
    }


    // Function to fetch specific synthesis stage data based on project_id
    // app/Http/Controllers/SynthesisStageController.php

    public function getSynthesisStagesByProject($projectId)
    {
        $stages = \DB::table('synthesis_stages')
            ->leftJoin('stage_comments', 'synthesis_stages.id', '=', 'stage_comments.stage_id')
            ->leftJoin('manager_approval', 'synthesis_stages.id', '=', 'manager_approval.stage_id')
            ->where('synthesis_stages.project_id', $projectId)
            ->select(
                'synthesis_stages.*',
                'stage_comments.comment',
                'manager_approval.manager_approval_status',
                'manager_approval.comments as approval_comments',
                'synthesis_stages.reaction',
                'synthesis_stages.scale',
                'synthesis_stages.experimental_conditions',
                'synthesis_stages.results',
                'synthesis_stages.yield',
                'synthesis_stages.data'
            )
            ->get();

        return response()->json([
            'stages' => $stages,
        ]);
    }

    /**
     * Show the synthesis stages for a specific project.
     *
     * @param  int  $projectId
     * @return \Illuminate\View\View
     */
    // Display synthesis stages and their details for a project
    public function showSynthesisStages($projectId)
    {
        // Fetch the project
        $project = ManufacturingProject::findOrFail($projectId);

        // Fetch synthesis stages, comments, and approvals for the project
        $stages = SynthesisStage::where('project_id', $projectId)->get();
        $stageIds = $stages->pluck('id');  // Get the stage IDs
        $comments = StageComment::whereIn('stage_id', $stages->pluck('id'))->get();
        $approvals = ManagerApproval::whereIn('stage_id', $stageIds)->get();

        return response()->json([
            'project' => $project,
            'stages' => $stages,
            'comments' => $comments,
            'approvals' => $approvals
        ]);
    }


    // Add a comment to a stage
    public function addComment(Request $request, $projectId, $stageId)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $comment = StageComment::create([
            'stage_id' => $stageId,
            'comment' => $request->comment
        ]);

        return response()->json(['message' => 'Comment added successfully!', 'comment' => $comment]);
    }

    // Handle manager approval of a stage
    public function approveStage(Request $request, $projectId, $stageId)
    {
        $request->validate([
            'comments' => 'nullable|string'
        ]);

        $approval = ManagerApproval::updateOrCreate(
            ['stage_id' => $stageId],
            ['approved' => true, 'comments' => $request->comments]
        );

        return response()->json(['message' => 'Stage Approved!', 'approval' => $approval]);
    }

    // Mark the synthesis as completed and move to inventory
    public function completeSynthesis(Request $request, $projectId)
    {
        $project = ManufacturingProject::findOrFail($projectId);
        $project->status = 'completed';
        $project->save();

        // You can use the inventory model to handle moving the product to inventory.
        // Assuming a quantity is given or calculated.
        $finalProduct = new Inventory([
            'product_name' => $project->project_name,
            'quantity' => 100  // You can adjust this based on your process
        ]);
        $finalProduct->save();

        return response()->json(['message' => 'Synthesis Complete and Product Moved to Inventory!']);
    }

    // Store the Route of Synthesis (ROS)
    public function storeROS(Request $request, $projectId)
    {
        $request->validate([
            'ros_document' => 'required|string'
        ]);

        $ros = RouteOfSynthesis::updateOrCreate(
            ['project_id' => $projectId],
            ['ros_document' => $request->ros_document]
        );

        return response()->json(['message' => 'Route of Synthesis saved successfully!', 'ros' => $ros]);
    }

    public function addSynthesisStage(Request $request, $projectId)
    {
        // Validate the request data
        $request->validate([
            'stage_name' => 'required|string|max:50',
            'batch_number' => 'required|integer',
            'progress' => 'required|numeric|min:0|max:100',
            'reaction' => 'nullable|string',
            'scale' => 'nullable|string',
            'experimental_conditions' => 'nullable|string',
            'results' => 'nullable|string',
            'yield' => 'nullable|numeric|min:0|max:100',
            'data' => 'nullable|string',
            'molecule_structure' => 'nullable|string',
        ]);

        $progress = (int) $request->input('progress');
        if ($progress === 100) {
            $project = ManufacturingProject::findOrFail($projectId);

            // Update the status
            $project->status = "Completed";
            $project->save();
        }

        // Create a new synthesis stage with the validated data
        $stage = SynthesisStage::create([
            'project_id' => $projectId,
            'stage_name' => $request->stage_name,
            'batch_number' => $request->batch_number,
            'progress' => $request->progress,
            'reaction' => $request->reaction,
            'scale' => $request->scale,
            'experimental_conditions' => $request->experimental_conditions,
            'results' => $request->results,
            'yield' => $request->yield,
            'data' => $request->data,
            'molecule_structure' => $request->molecule_structure,

        ]);

        // Return a success response
        return response()->json(['message' => 'Synthesis stage added successfully!', 'stage' => $stage]);
    }


    public function getSynthesisStages($projectId)
    {
        // Get all synthesis stages for the project
        $stages = SynthesisStage::where('project_id', $projectId)->get();

        return response()->json(['stages' => $stages]);
    }

    // public function requestManagerApproval(Request $request)
    // {
    //     // Check if the stage is T3
    //     $stage = SynthesisStage::where('project_id', $request->project_id)
    //         ->where('stage_name', 'T3')
    //         ->first();

    //     if ($stage && $stage->progress < 100) {
    //         // Check if the manager has already approved or not
    //         $approval = ManagerApproval::where('stage_id', $stage->id)->first();

    //         // If not approved yet, allow the manager to approve
    //         if (!$approval || $approval->approved == false) {
    //             // Update the approval request (this could be based on manager input, 
    //             // here we assume a request is being sent to initiate the approval process)
    //             $approval = ManagerApproval::updateOrCreate(
    //                 ['stage_id' => $stage->id],
    //                 ['approved' => false, 'comments' => 'Approval Pending']
    //             );

    //             // Return success message to the frontend
    //             return response()->json([
    //                 'status' => 'success',
    //                 'message' => 'Manager approval request has been sent for T3. Please wait for approval to proceed.'
    //             ]);
    //         }

    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Stage T3 is already approved, or the approval is pending.'
    //         ]);
    //     }

    //     return response()->json([
    //         'status' => 'error',
    //         'message' => 'Stage T3 not found, or synthesis is already complete.'
    //     ]);
    // }

    public function approveSynthesisStage(Request $request)
    {
        // Validate the request
        $request->validate([
            'stage_id' => 'required|exists:synthesis_stages,id',
            'approved' => 'required|boolean',
            'comments' => 'nullable|string',
        ]);

        // Find the manager approval record
        $approval = ManagerApproval::where('stage_id', $request->stage_id)->first();

        if ($approval) {
            // Update approval status
            $approval->approved = $request->approved;
            $approval->comments = $request->comments;
            $approval->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Manager approval has been updated.'
            ]);
        }

        return response()->json([
            'status' => 'error',
            'message' => 'No approval request found for the given stage.'
        ]);
    }

    public function requestManagerApproval(Request $request)
    {
        // Validate the incoming data
        $validator = Validator::make($request->all(), [
            'project_id' => 'required|exists:manufacturing_projects,id',
            'stage_id' => 'required|exists:synthesis_stages,id',
            'comments' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => 'Validation failed', 'errors' => $validator->errors()], 422);
        }

        try {
            // Create a new ManagerApproval record
            $managerApproval = new ManagerApproval([
                'project_id' => $request->project_id,
                'stage_id' => $request->stage_id,
                'manager_approval_status' => 2,
                'comments' => $request->comments,
            ]);

            $managerApproval->save();

            // Respond back with a success message
            return response()->json(['message' => 'Manager approval request has been submitted successfully.']);
        } catch (\Exception $e) {
            // Log the error
            Log::error('Error saving manager approval request: ' . $e->getMessage());

            // Respond with a generic error message
            return response()->json(['message' => 'An error occurred while processing your request. Please try again later.'], 500);
        }
    }

    public function getProjectDetailsForManager(Request $request)
    {
        $projects = ManufacturingProject::query();

        if ($request->has('status')) {
            $projects->where('allocation_status', "Allocated");
        }

        $projects = $projects->leftJoin('inventory_allocations', 'manufacturing_projects.id', '=', 'inventory_allocations.project_id')
            ->leftJoin('synthesis_stages', 'manufacturing_projects.id', '=', 'synthesis_stages.project_id')
            ->leftJoin('manager_approval', 'synthesis_stages.id', '=', 'manager_approval.stage_id')
            ->select(
                'manufacturing_projects.id as project_id',
                'manufacturing_projects.project_name', // Assuming you have a project_name field
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.stage_name ORDER BY synthesis_stages.id ASC) as stage_names'),
                DB::raw('GROUP_CONCAT(DISTINCT manager_approval.manager_approval_status ORDER BY synthesis_stages.id ASC) as approval_statuses')
            )
            ->where('manufacturing_projects.allocation_status', "Allocated")
            ->groupBy('manufacturing_projects.id')
            ->paginate(10);

        $projectsData = collect($projects->items())->map(function ($project) {
            return [
                'project_id' => $project->project_id,
                'project_name' => $project->project_name,
                'stage_names' => $project->stage_names,
                'manager_approval_status' => $project->approval_statuses,
            ];
        });

        return response()->json([
            'draw' => $request->input('draw'),
            'recordsTotal' => $projects->total(),
            'recordsFiltered' => $projects->count(),
            'data' => $projectsData,
        ]);
    }

    public function getStagesForApproval($project_id)
    {
        // Fetch all stages for the project
        $stages = SynthesisStage::where('synthesis_stages.project_id', $project_id) // Specify the table
            ->leftJoin('manager_approval', 'synthesis_stages.id', '=', 'manager_approval.stage_id')
            ->select('synthesis_stages.*', 'manager_approval.manager_approval_status', 'manager_approval.comments')
            ->get();

        return response()->json([
            'stages' => $stages
        ]);
    }



    public function submitApproval(Request $request)
    {
        // Get input data
        $projectId = $request->input('projectId');
        $status = $request->input('action');
        $comment = $request->input('comment');

        // Fetch the project approval record
        $approval = ManagerApproval::where('project_id', $projectId)->first();

        if (!$approval) {
            return response()->json(['success' => false, 'message' => 'Approval record not found.']);
        }

        // Update the approval status and comment
        $approval->manager_approval_status = $status;
        $approval->comments = $comment;
        $approval->save();

        // Return success response
        return response()->json(['success' => true, 'message' => 'Approval status updated successfully']);
    }

    public function getReadyToDispatch()
    {
        $projects = DB::table('manufacturing_projects')
            ->leftJoin('inventory_allocations', 'manufacturing_projects.id', '=', 'inventory_allocations.project_id')
            ->leftJoin('synthesis_stages', 'manufacturing_projects.id', '=', 'synthesis_stages.project_id')
            ->leftJoin('stage_comments', 'synthesis_stages.id', '=', 'stage_comments.stage_id')
            ->leftJoin('manager_approval', 'synthesis_stages.id', '=', 'manager_approval.stage_id')
            ->leftJoin('order_shipping_details', 'manufacturing_projects.id', '=', 'order_shipping_details.project_id')
            ->select(
                'manufacturing_projects.id as project_id',
                'manufacturing_projects.project_name',
                'manufacturing_projects.order_id',
                'manufacturing_projects.status as project_status',
                'order_shipping_details.dispatch_status as dispatch_status',
                'manufacturing_projects.created_at as project_created_at',
                'manufacturing_projects.updated_at as project_updated_at',
                'manufacturing_projects.allocation_status',
                DB::raw('SUM(inventory_allocations.quantity) as total_allocated'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.stage_name ORDER BY synthesis_stages.id ASC) as stage_names'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.batch_number ORDER BY synthesis_stages.id ASC) as batch_numbers'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.progress ORDER BY synthesis_stages.id ASC) as progresses'),
                DB::raw('GROUP_CONCAT(DISTINCT synthesis_stages.id ORDER BY synthesis_stages.id ASC) as stage_ids'),
                DB::raw('GROUP_CONCAT(DISTINCT stage_comments.comment ORDER BY synthesis_stages.id ASC) as stage_comments'),
                DB::raw('GROUP_CONCAT(DISTINCT manager_approval.manager_approval_status ORDER BY synthesis_stages.id ASC) as approval_statuses'),
                DB::raw('GROUP_CONCAT(DISTINCT manager_approval.comments ORDER BY synthesis_stages.id ASC) as approval_comments')
            )
            ->where('manufacturing_projects.allocation_status', "Allocated")
            ->where(function ($query) {
                $query->where('synthesis_stages.progress', 100);
            })
            ->groupBy('manufacturing_projects.id')
            ->paginate(10);

        return response()->json($projects);
    }

    public function getFullProjectDetails($projectId)
    {
        // Correct the query to return full details for each synthesis stage and include dispatch status
        $query = "
            SELECT
                p.id as project_id,
                p.project_name,
                p.order_id,
                p.status as project_status,
                p.created_at as project_created_at,
                p.updated_at as project_updated_at,
                p.allocation_status,
                p.roadmap,
                d.country as dispatch_country,
                d.medium as dispatch_medium,
                d.freight_terms as dispatch_freight_terms,
                i.quantity as total_allocated,
                i.location as inventory_location,
                i.unit_price as inventory_unit_price,
                ss.id as stage_id,
                ss.stage_name,
                ss.batch_number,
                ss.progress,
                sc.comment as stage_comment,
                ma.manager_approval_status,
                ma.comments as approval_comment,
                osd.dispatch_status  as dispatch_status
            FROM
                manufacturing_projects p
            LEFT JOIN
                client_dispatch_info d ON p.id = d.project_id
            LEFT JOIN
                inventory_allocations i ON p.id = i.project_id
            LEFT JOIN
                synthesis_stages ss ON p.id = ss.project_id
            LEFT JOIN
                stage_comments sc ON ss.id = sc.stage_id
            LEFT JOIN
                manager_approval ma ON ss.id = ma.stage_id
            LEFT JOIN
                order_shipping_details osd ON p.id = osd.project_id  -- Join to get the dispatch status
            WHERE
                p.id = :projectId
        ";

        // Execute the query with the projectId parameter
        $project = DB::select($query, ['projectId' => $projectId]);

        if (empty($project)) {
            return response()->json(['status' => 'error', 'message' => 'Project not found'], 404);
        }

        // Format the result to group by project data and synthesis stages
        $projectData = [
            'project_name' => $project[0]->project_name,
            'project_status' => $project[0]->project_status,
            'order_id' => $project[0]->order_id,
            'created_at' => $project[0]->project_created_at,
            'dispatch_country' => $project[0]->dispatch_country,
            'dispatch_medium' => $project[0]->dispatch_medium,
            'dispatch_freight_terms' => $project[0]->dispatch_freight_terms,
            'total_allocated' => $project[0]->total_allocated,
            'inventory_location' => $project[0]->inventory_location,
            'inventory_unit_price' => $project[0]->inventory_unit_price,
            'dispatch_status' => $project[0]->dispatch_status,  // Added dispatch_status to project data
            'synthesis_stages' => [],
        ];

        // Iterate through synthesis stages and group them
        foreach ($project as $stage) {
            $projectData['synthesis_stages'][] = [
                'stage_id' => $stage->stage_id,
                'stage_name' => $stage->stage_name,
                'batch_number' => $stage->batch_number,
                'progress' => $stage->progress,
                'stage_comment' => $stage->stage_comment,
                'manager_approval_status' => $stage->manager_approval_status,
                'approval_comment' => $stage->approval_comment,
            ];
        }

        return response()->json([
            'status' => 'success',
            'data' => $projectData,
        ]);
    }


    public function OrderShippingDetails(Request $request)
    {
        // Validate the incoming request data
        $validatedData = $request->validate([
            'dispatch_date' => 'required|date',
            'invoice_number' => 'nullable|string|max:50',
            'tracking_number' => 'nullable|string|max:255',
            'dispatch_mode' => 'required|string|max:50',
            'dispatch_status' => 'required|in:Pending,Shipped,Delivered,Returned,In Transit,Out for Delivery,Dispatched to Warehouse,Canceled,On Hold,Ready for Dispatch,In Customs,Shipment Delayed,Scheduled for Pickup,Failed Delivery Attempt,Partial Delivery,Returned to Sender,Delivered to Pickup Point,Awaiting Customer Confirmation,Dispatch Failed,Dispatched to Third Party',
            'total_dispatch_weight' => 'nullable|numeric|min:0',
            'total_dispatch_value' => 'nullable|numeric|min:0',
            'insurance_required' => 'nullable|boolean',
            'insurance_provider' => 'nullable|string|max:255',
            'dispatch_priority' => 'nullable|string|max:50',
            'estimated_delivery_date' => 'nullable|date',
            'proof_of_delivery' => 'nullable|boolean',
            'dispatch_method' => 'nullable|string|max:50',
            'port_of_origin' => 'nullable|string|max:255',
            'port_of_destination' => 'nullable|string|max:255',
            'dispatch_notes' => 'nullable|string',
        ]);

        // Create a new dispatch record
        try {
            $dispatch = new OrderShippingDetails();
            $dispatch->project_id = $request->input('projectId_for_dispatch');
            $dispatch->dispatch_date = $validatedData['dispatch_date'];
            $dispatch->invoice_number = $validatedData['invoice_number'];
            $dispatch->tracking_number = $validatedData['tracking_number'];
            $dispatch->dispatch_mode = $validatedData['dispatch_mode'];
            $dispatch->dispatch_status = $validatedData['dispatch_status'];
            $dispatch->total_dispatch_weight = $validatedData['total_dispatch_weight'];
            $dispatch->total_dispatch_value = $validatedData['total_dispatch_value'];
            $dispatch->insurance_required = $validatedData['insurance_required'] ?? 0;
            $dispatch->insurance_provider = $validatedData['insurance_provider'];
            $dispatch->dispatch_priority = $validatedData['dispatch_priority'];
            $dispatch->estimated_delivery_date = $validatedData['estimated_delivery_date'];
            $dispatch->proof_of_delivery = $validatedData['proof_of_delivery'] ?? 0;
            $dispatch->dispatch_method = $validatedData['dispatch_method'];
            $dispatch->port_of_origin = $validatedData['port_of_origin'];
            $dispatch->port_of_destination = $validatedData['port_of_destination'];
            $dispatch->dispatch_notes = $validatedData['dispatch_notes'];

            // Save the dispatch information
            $dispatch->save();

            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }


    public function purchaseInventory(Request $request)
    {
        // Validate incoming data
        $validatedData = $request->validate([
            'material_name' => 'required|string',
            'material_code' => 'required|string',
            'quantity' => 'required|integer|min:1',
            'vendors' => 'required|array|min:1', // Vendors should be selected
        ]);

        // Find the material in the inventory
        $inventory = ManufacturingInventory::where('material_code', $request->material_code)->first();

        if (!$inventory) {
            return response()->json(['success' => false, 'message' => 'Material not found in inventory.']);
        }

        // Check if stock is available
        if (!$inventory->isStockAvailable($request->quantity)) {
            return response()->json(['success' => false, 'message' => 'Insufficient stock available.']);
        }

        // Reduce the stock in inventory
        $inventory->reduceStock($request->quantity);

        // Optionally, log the purchase or associate the vendors (this part can be extended)
        // Here we're assuming a simple log or transaction for the purchase

        // For now, let's just simulate storing vendors (this can be extended to a real relationship)
        // foreach ($request->vendors as $vendorId) {
        //     PurchaseLog::create([
        //         'material_code' => $request->material_code,
        //         'vendor_id' => $vendorId,
        //         'quantity' => $request->quantity,
        //     ]);
        // }

        // Respond with success
        return response()->json(['success' => true, 'message' => 'Purchase processed successfully.']);
    }

    public function getDispatchDetails(Request $request)
    {
        // Extract the necessary parameters from the DataTables request
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        $search = $request->input('search.value', ''); // Search value

        // Build the query with pagination and search
        $query = OrderShippingDetails::query();

        // Apply search filter if there is any search term
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                    ->orWhere('tracking_number', 'like', "%{$search}%")
                    ->orWhere('dispatch_mode', 'like', "%{$search}%")
                    ->orWhere('dispatch_status', 'like', "%{$search}%")
                    ->orWhere('dispatch_priority', 'like', "%{$search}%");
            });
        }

        // Apply sorting if there are order parameters from DataTables (DataTables uses 'order' param)
        $orderColumnIndex = $request->input('order.0.column');
        $orderDirection = $request->input('order.0.dir', 'asc'); // default is ascending

        $columns = ['id', 'project_id', 'dispatch_date', 'invoice_number', 'tracking_number', 'dispatch_mode', 'dispatch_status', 'total_dispatch_weight', 'total_dispatch_value', 'insurance_required', 'insurance_provider', 'dispatch_priority', 'estimated_delivery_date'];

        if ($orderColumnIndex !== null) {
            $query->orderBy($columns[$orderColumnIndex], $orderDirection);
        }

        // Get total records without pagination for 'recordsTotal'
        $totalRecords = OrderShippingDetails::count();

        // Get filtered records with pagination
        $dispatchDetails = $query->skip($start)->take($length)->get();

        // Format the response to match DataTables expected format
        return response()->json([
            'draw' => $request->input('draw'),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $dispatchDetails->count(),
            'data' => $dispatchDetails
        ]);
    }
}
